/*:
 * @target MZ
 * @plugindesc v1.3 右上ツールチップ（多言語OK）。フォント固定＆縦位置補正（TextYOffset/LinePad）を追加
 * @author HS
 *
 * @param DefaultVarId
 * @type variable
 * @default 0
 * @text 既定の文字列変数ID
 *
 * @param ImportantSwitchId
 * @type switch
 * @default 0
 * @text 重要シーン切替スイッチID
 * @desc ON中は自動的に非表示（HS_FastModeと同IDにすると連動）
 *
 * @param Anchor
 * @type select
 * @option TopRight
 * @option TopLeft
 * @option BottomRight
 * @option BottomLeft
 * @option Custom
 * @default TopRight
 * @text 位置
 *
 * @param MarginX
 * @type number
 * @min 0
 * @default 24
 * @text マージンX
 *
 * @param MarginY
 * @type number
 * @min 0
 * @default 16
 * @text マージンY
 *
 * @param MaxWidth
 * @type number
 * @min 0
 * @default 420
 * @text 最大幅(px, 0=自動)
 *
 * @param FontSize
 * @type number
 * @min 10
 * @default 20
 * @text フォントサイズ
 *
 * @param LinePad
 * @type number
 * @min 0
 * @default 6
 * @text 行パディング(フォント+余白)
 * @desc 行の高さ=FontSize+LinePad
 *
 * @param TextYOffset
 * @type number
 * @min -32
 * @max 32
 * @default 0
 * @text テキストY補正
 * @desc 文字の描画位置を上下に微調整（見た目で下寄りなら-2〜-4推奨）
 *
 * @param WindowOpacity
 * @type number
 * @min 0
 * @max 255
 * @default 210
 * @text ウィンドウ不透明度
 *
 * @param BackOpacity
 * @type number
 * @min 0
 * @max 255
 * @default 96
 * @text 背景不透明度
 *
 * @param AutoHideDefault
 * @type number
 * @min 0
 * @default 300
 * @text 既定の自動非表示(フレーム,0=無し)
 *
 * @param ParseEscape
 * @type boolean
 * @on はい
 * @off いいえ
 * @default true
 * @text 制御文字(\C,\I,\V)を解釈
 *
 * @param PersistAcrossScenes
 * @type boolean
 * @on はい
 * @off いいえ
 * @default true
 * @text シーン跨ぎで維持
 *
 * @command ShowFromVariable
 * @text 変数から表示
 * @arg varId
 * @type variable
 * @default 0
 * @text 文字列変数ID
 * @arg autohide
 * @type number
 * @default 300
 * @text 自動非表示(0=無し)
 *
 * @command ShowText
 * @text テキストを表示
 * @arg text
 * @type string
 * @default
 * @text 表示文字列
 * @arg autohide
 * @type number
 * @default 300
 * @text 自動非表示(0=無し)
 *
 * @command HideTooltip
 * @text 非表示にする
 *
 * @command SetAnchor
 * @text 位置を変更
 * @arg anchor
 * @type select
 * @option TopRight
 * @option TopLeft
 * @option BottomRight
 * @option BottomLeft
 * @option Custom
 * @default TopRight
 * @text 位置
 * @arg marginX
 * @type number
 * @default 24
 * @text マージンX
 * @arg marginY
 * @type number
 * @default 16
 * @text マージンY
 *
 * @command BindVariable
 * @text 変数にバインド（常駐）
 * @arg varId
 * @type variable
 * @default 0
 * @text 文字列変数ID
 */
(() => {
  "use strict";
  const PN = "HS_TooltipCE";
  const P  = PluginManager.parameters(PN);

  let DEF_VAR   = Number(P["DefaultVarId"] || 0);
  const IMP_SW  = Number(P["ImportantSwitchId"] || 0);
  let ANCHOR    = String(P["Anchor"] || "TopRight");
  let MARGIN_X  = Number(P["MarginX"] || 24);
  let MARGIN_Y  = Number(P["MarginY"] || 16);
  let MAX_W     = Math.max(0, Number(P["MaxWidth"] || 420));
  let FONT_SIZE = Math.max(10, Number(P["FontSize"] || 20));
  let LINE_PAD  = Math.max(0, Number(P["LinePad"] || 6));
  let TEXT_DY   = Number(P["TextYOffset"] || 0);
  let W_OPA     = Math.min(255, Math.max(0, Number(P["WindowOpacity"] || 210)));
  let B_OPA     = Math.min(255, Math.max(0, Number(P["BackOpacity"] || 96)));
  let AUTO_DEF  = Math.max(0, Number(P["AutoHideDefault"] || 300));
  const ESCAPE  = String(P["ParseEscape"] || "true") === "true";
  const PERSIST = String(P["PersistAcrossScenes"] || "true") === "true";

  const HSTip = { text:"", boundVarId:0, autohide:0, visible:false, needRefresh:false };

  class Window_HSTooltip extends Window_Base {
    initialize(rect) {
      super.initialize(rect);
      this.openness = 0;
      this.contentsOpacity = 255;
      this._lastDrawn = "";
    }
    // ★ このウィンドウだけ、フォントサイズを強制保持
    resetFontSettings() {
      Window_Base.prototype.resetFontSettings.call(this);
      this.contents.fontSize = FONT_SIZE;
    }
    lineHeight() { return Math.max(16, FONT_SIZE + LINE_PAD); }
    standardFontSize(){ return FONT_SIZE; }
    update() {
      super.update();
      const blocked = IMP_SW > 0 && $gameSwitches && $gameSwitches.value(IMP_SW);
      const shouldShow = HSTip.visible && !blocked;
      if (shouldShow && this.openness < 255) this.open();
      if (!shouldShow && this.openness > 0) this.close();

      if (HSTip.boundVarId > 0) {
        const v = String($gameVariables.value(HSTip.boundVarId) ?? "");
        if (v !== HSTip.text) { HSTip.text = v; HSTip.needRefresh = true; }
      }
      if (shouldShow && HSTip.autohide > 0) {
        if (--HSTip.autohide <= 0) { HSTip.visible = false; HSTip.autohide = 0; }
      }
      if (HSTip.needRefresh || (shouldShow && this._lastDrawn !== HSTip.text)) this.refresh();
      this.updateAnchor();
    }
    convertIfNeeded(t){ return ESCAPE ? this.convertEscapeCharacters(t || "") : (t || ""); }
    textLines(){ return String(this.convertIfNeeded(HSTip.text)).split(/\r?\n/); }
    calcNeededSize(){
      const pad = this.padding;
      let w = 0;
      for (const ln of this.textLines()){
        // resetFontSettings() がFONT_SIZEを充ててくれる
        const ts = this.textSizeEx(ln);
        w = Math.max(w, ts.width + 8);
      }
      if (MAX_W > 0) w = Math.min(w, MAX_W);
      const h = this.lineHeight() * Math.max(1, this.textLines().length) + pad * 2;
      return new Rectangle(0,0, Math.max(120, w + pad * 2), Math.max(this.lineHeight()+pad*2, h));
    }
    refresh(){
      this.contents.clear();
      this.opacity = W_OPA;
      this.backOpacity = B_OPA;
      const need = this.calcNeededSize();
      this.move(this.x, this.y, need.width, need.height);
      this.createContents();
      this.resetFontSettings();
      let y = this.padding + TEXT_DY;
      for (const ln of this.textLines()){
        this.drawTextEx(ln, this.padding, y, this.contentsWidth());
        y += this.lineHeight();
      }
      this._lastDrawn = HSTip.text;
      HSTip.needRefresh = false;
    }
    updateAnchor(){
      const bw = Graphics.boxWidth, bh = Graphics.boxHeight;
      const w = this.width, h = this.height;
      let x = this.x, y = this.y;
      switch (ANCHOR){
        case "TopRight":    x = bw - w - MARGIN_X; y = MARGIN_Y; break;
        case "TopLeft":     x = MARGIN_X;          y = MARGIN_Y; break;
        case "BottomRight": x = bw - w - MARGIN_X; y = bh - h - MARGIN_Y; break;
        case "BottomLeft":  x = MARGIN_X;          y = bh - h - MARGIN_Y; break;
        case "Custom": default: break;
      }
      this.x = Math.max(0, x);
      this.y = Math.max(0, y);
    }
  }

  // --- 安全生成（v1.2と同じ）
  function ensureTooltipWindow(){
    const scene = SceneManager._scene;
    if (!scene) return false;
    if (scene._hsTooltip && !scene._hsTooltip._destroyed) return true;

    const createNow = () => {
      const rect = new Rectangle(0,0,320,64);
      const win = new Window_HSTooltip(rect);
      scene._hsTooltip = win;
      if (scene._windowLayer && scene.addWindow) scene.addWindow(win);
      else if (scene.addChild) scene.addChild(win);
      else { scene._hsTooltip = null; return false; }
      win.close();
      return true;
    };
    if (!scene._windowLayer) {
      scene._hsTipPending = true;
      const _update = scene.update;
      scene.update = function(){
        _update.call(this);
        if (this._hsTipPending && this._windowLayer){
          this._hsTipPending = false;
          createNow();
        }
      };
      return true;
    }
    return createNow();
  }

  function installToScene(SceneKlass){
    const _start = SceneKlass.prototype.start;
    SceneKlass.prototype.start = function(){
      _start.call(this);
      if (PERSIST || this instanceof Scene_Map){ ensureTooltipWindow(); }
    };
  }
  installToScene(Scene_Map);
  installToScene(Scene_Battle);
  installToScene(Scene_MenuBase);

  // ---- Plugin Commands
  PluginManager.registerCommand(PN, "ShowFromVariable", args => {
    const varId = Number(args.varId || DEF_VAR || 0);
    const auto  = Math.max(0, Number(args.autohide || AUTO_DEF));
    const txt   = varId > 0 ? String($gameVariables.value(varId) ?? "") : "";
    ensureTooltipWindow();
    if (SceneManager._scene && SceneManager._scene._hsTooltip){
      HSTip.text = txt; HSTip.boundVarId = 0; HSTip.autohide = auto;
      HSTip.visible = !!txt; HSTip.needRefresh = true;
    }
  });

  PluginManager.registerCommand(PN, "ShowText", args => {
    const txt  = String(args.text || "");
    const auto = Math.max(0, Number(args.autohide || AUTO_DEF));
    ensureTooltipWindow();
    if (SceneManager._scene && SceneManager._scene._hsTooltip){
      HSTip.text = txt; HSTip.boundVarId = 0; HSTip.autohide = auto;
      HSTip.visible = !!txt; HSTip.needRefresh = true;
    }
  });

  PluginManager.registerCommand(PN, "HideTooltip", () => {
    if (SceneManager._scene && SceneManager._scene._hsTooltip){
      HSTip.visible = false; HSTip.autohide = 0;
    }
  });

  PluginManager.registerCommand(PN, "SetAnchor", args => {
    ANCHOR   = String(args.anchor || ANCHOR);
    MARGIN_X = Number(args.marginX || MARGIN_X);
    MARGIN_Y = Number(args.marginY || MARGIN_Y);
    HSTip.needRefresh = true;
  });

  PluginManager.registerCommand(PN, "BindVariable", args => {
    const varId = Number(args.varId || 0);
    ensureTooltipWindow();
    if (SceneManager._scene && SceneManager._scene._hsTooltip){
      HSTip.boundVarId = varId; HSTip.visible = true; HSTip.autohide = 0;
      HSTip.needRefresh = true;
    }
  });
})();

